/*
 * NetHostFS by Dark_AleX
 *
 * Inspired by psplink
*/


#include <pspsdk.h>
#include <pspkernel.h>
#include <unistd.h>
#include <pspnet.h>
#include <pspnet_inet.h>
#include <pspnet_apctl.h>
#include <pspnet_resolver.h>
#include <arpa/inet.h>
#include <sys/select.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>

#include "nethostfs.h"

PSP_MODULE_INFO("NetHostFS", 0x1000, 1, 1);
PSP_HEAP_SIZE_KB(0);

int driverInited=0;
int driverInitError=0;

#define ARG_MAX	10

#define SEND_INTEGER(x) \
	if (sceNetInetSend(server, &x, 4, 0) != 4) \
		return -1; 	

#define RECV_INTEGER(x) \
	if (sceNetInetRecv(server, x, 4, 0) != 4) \
		return -1; 	

#define SEND_COMMAND(x) \
	cmd = x; \
	if (sceNetInetSend(server, &cmd, 4, 0) != 4) \
		return -1; 	

#define SEND_COMMAND_WITH_PARAMS(x) \
	cmd_params.cmd = x; \
	if (sceNetInetSend(server, &cmd_params, sizeof(cmd_params), 0) != sizeof(cmd_params)) \
		return -1; 	

#define RETURN_FROM_HOST() \
	if (sceNetInetRecv(server, &res, 4, 0) != 4) \
		return -1; \
	return res

#define RECV_RESULT() \
	if (sceNetInetRecv(server, &result, sizeof(result), 0) != sizeof(result)) \
		return -1; 	

int server=-1;
char ip_address[16];


int connect_to_apctl(int config)
{
	int err;
	int stateLast = -1;

	err = sceNetApctlConnect(config);
	if (err < 0)
	{
		return -1;
	}

	while (1)
	{
		int state;

		err = sceNetApctlGetState(&state);
		if (err < 0)
		{
			break;
		}
		if (state > stateLast)
		{
			//printf("  connection state %d of 4\n", state);
			stateLast = state;
		}
		if (state == 4)
			break;  // connected with static IP

		// wait a little before polling again
		sceKernelDelayThread(50*1000); // 50ms
	}
	
	if(err < 0)
	{
		return -1;
	}

	//printf("connected!\n");

	return 0;
}

int nethostfs_init(PspIoDrvArg* arg)
{
	struct sockaddr_in addr;
	int cmd, flag, res;
	
	if (connect_to_apctl(1) < 0)
	{
		return -1;
	}

	server = sceNetInetSocket(PF_INET, SOCK_STREAM, 0);

	if (server < 0)
	{
		return -1;
	}

	flag = 1;
	sceNetInetSetsockopt(server, SOL_TCP, TCP_NODELAY, &flag, sizeof(flag));

	addr.sin_family = AF_INET;
    addr.sin_port = htons(7513); 
    addr.sin_addr.s_addr = sceNetInetInetAddr(ip_address);

	if (sceNetInetConnect(server, (struct sockaddr *)&addr, sizeof(addr)) < 0)
	{
		return -1;
	}
	
	SEND_COMMAND(NET_HOSTFS_CMD_HELLO);
	RECV_INTEGER(&cmd);	

	if (cmd != NET_HOSTFS_CMD_HELLO)
	{
		return -1;
	}

	SEND_COMMAND(NET_HOSTFS_CMD_IOINIT);
	RETURN_FROM_HOST();	
}

int nethostfs_exit(PspIoDrvArg* arg)
{
	int cmd;
	
	SEND_COMMAND(NET_HOSTFS_CMD_IOEXIT);
	//sceNetInetClose(server);
	//server = -1;

	return 0;
}

int nethostfs_open(PspIoDrvFileArg *arg, char *file, int flags, SceMode mode)
{
	PACKEDSTRUCT
	{
		int cmd;
		IO_OPEN_PARAMS params;		
	} cmd_params;

	int result;

	strncpy(cmd_params.params.file, file, 256);
	cmd_params.params.fs_num = arg->fs_num;
	cmd_params.params.flags = flags;
	cmd_params.params.mode = mode;

	SEND_COMMAND_WITH_PARAMS(NET_HOSTFS_CMD_IOOPEN);	
	RECV_RESULT();

	if (result < 0)
		return -1;
	else
		arg->arg = (void *)result;
	
	return 0;
}

int nethostfs_close(PspIoDrvFileArg *arg)
{
	PACKEDSTRUCT
	{
		int cmd;
		int fd;
	} cmd_params;

	int res;

	cmd_params.fd = (int)arg->arg;	

	SEND_COMMAND_WITH_PARAMS(NET_HOSTFS_CMD_IOCLOSE);
	RETURN_FROM_HOST();
}

int nethostfs_read(PspIoDrvFileArg *arg, char *data, int len)
{
	PACKEDSTRUCT
	{
		int cmd;
		IO_READ_PARAMS params;
	} cmd_params;

	int bytesread;

	if (len < 0)
		return -1;
	else if (len == 0)
		return 0;

	cmd_params.params.fd = (int)arg->arg;
	cmd_params.params.len = len;

	SEND_COMMAND_WITH_PARAMS(NET_HOSTFS_CMD_IOREAD);
	RECV_INTEGER(&bytesread);

	if (bytesread > len) // !?
		return -1;

	if (bytesread > 0)
	{
		char *pbuf = data;
		int received = 0, x;

		while (received < bytesread)
		{
			int blocksize;

			if ((bytesread - received) >= 2048)
				blocksize = 2048;
			else
				blocksize = bytesread - received;

			x = sceNetInetRecv(server, pbuf, blocksize, 0);

			if (x <= 0)
				return -1;

			received += x;
			pbuf += x;
		}
	}

	return bytesread;
}

int nethostfs_write(PspIoDrvFileArg *arg, const char *data, int len)
{
	PACKEDSTRUCT
	{
		int cmd;
		IO_WRITE_PARAMS params;
	} cmd_params;

	int x, sent, res;
	char *pbuf;

	if (len < 0)
		return -1;
	else if (len == 0)
		return 0;

	cmd_params.params.fd = (int)arg->arg;
	cmd_params.params.len = len;

	SEND_COMMAND_WITH_PARAMS(NET_HOSTFS_CMD_IOWRITE);

	sent = 0;
	pbuf = (char *)data;

	while (sent < len)
	{
		int blocksize;

		if ((len - sent) >= 2048)
			blocksize = 2048;
		else
			blocksize = len - sent;

		x = sceNetInetSend(server, pbuf, blocksize, 0);

		if (x <= 0)
			return -1;

		sent += x;
		pbuf += x;
	}

	RETURN_FROM_HOST();
}

SceOff nethostfs_lseek(PspIoDrvFileArg *arg, SceInt64 offs, int whence)
{
	PACKEDSTRUCT
	{
		int cmd;
		IO_LSEEK_PARAMS params;
	} cmd_params;

	SceOff res=0;	

	cmd_params.params.fd = (int)arg->arg;
	cmd_params.params.offset = offs;
	cmd_params.params.whence = whence;

	SEND_COMMAND_WITH_PARAMS(NET_HOSTFS_CMD_IOLSEEK);
	RETURN_FROM_HOST();
}

int nethostfs_ioctl(PspIoDrvFileArg *arg, unsigned int cmd, void *indata, int inlen, void *outdata, int outlen)
{
	// Not implemented
	return 0;
}

int nethostfs_remove(PspIoDrvFileArg *arg, const char *name)
{
	PACKEDSTRUCT
	{
		int cmd;
		IO_REMOVE_PARAMS params;
	} cmd_params;

	int res;

	strncpy(cmd_params.params.file, name, 256);
	cmd_params.params.fs_num = arg->fs_num;

	SEND_COMMAND_WITH_PARAMS(NET_HOSTFS_CMD_IOREMOVE);
	RETURN_FROM_HOST();
}

int nethostfs_mkdir(PspIoDrvFileArg *arg, const char *name, SceMode mode)
{
	PACKEDSTRUCT
	{
		int cmd;
		IO_MKDIR_PARAMS params;
	} cmd_params;

	int res;

	strncpy(cmd_params.params.dir, name, 256);
	cmd_params.params.fs_num = arg->fs_num;
	cmd_params.params.mode = mode;

	SEND_COMMAND_WITH_PARAMS(NET_HOSTFS_CMD_IOMKDIR);
	RETURN_FROM_HOST();
}

int nethostfs_rmdir(PspIoDrvFileArg *arg, const char *name)
{
	PACKEDSTRUCT
	{
		int cmd;
		IO_RMDIR_PARAMS params;
	} cmd_params;

	int res;

	strncpy(cmd_params.params.dir, name, 256);
	cmd_params.params.fs_num = arg->fs_num;
	
	SEND_COMMAND_WITH_PARAMS(NET_HOSTFS_CMD_IORMDIR);
	RETURN_FROM_HOST();
}

int nethostfs_dopen(PspIoDrvFileArg *arg, const char *dirname)
{
	PACKEDSTRUCT
	{
		int cmd;
		IO_DOPEN_PARAMS params;
	} cmd_params;

	int res;

	strncpy(cmd_params.params.dir, dirname, 256);
	cmd_params.params.fs_num = arg->fs_num;

	SEND_COMMAND_WITH_PARAMS(NET_HOSTFS_CMD_IODOPEN);
	RETURN_FROM_HOST();
}

int nethostfs_dclose(PspIoDrvFileArg *arg)
{
	PACKEDSTRUCT
	{
		int cmd;
		int fd;
	} cmd_params;

	int res;

	cmd_params.fd = (int)arg->arg;

	SEND_COMMAND_WITH_PARAMS(NET_HOSTFS_CMD_IODCLOSE);
	RETURN_FROM_HOST();
}

int nethostfs_dread(PspIoDrvFileArg *arg, SceIoDirent *dir)
{
	PACKEDSTRUCT
	{
		int cmd;
		int fd;
	} cmd_params;

	IO_DREAD_RESULT result;

	cmd_params.fd = (int)arg->arg;
	
	SEND_COMMAND_WITH_PARAMS(NET_HOSTFS_CMD_IODREAD);
	RECV_RESULT();

	if (result.res > 0)
	{
		memcpy(dir, &result.entry, sizeof(SceIoDirent));
	}

	return result.res;
}

int nethostfs_getstat(PspIoDrvFileArg *arg, const char *file, SceIoStat *stat)
{
	PACKEDSTRUCT
	{
		int cmd;
		IO_GETSTAT_PARAMS params;
	} cmd_params;

	IO_GETSTAT_RESULT result;

	strncpy(cmd_params.params.file, file, 256);
	cmd_params.params.fs_num = arg->fs_num;

	SEND_COMMAND_WITH_PARAMS(NET_HOSTFS_CMD_IOGETSTAT);
	RECV_RESULT();

	if (result.res >= 0)
	{
		memcpy(stat, &result.stat, sizeof(SceIoStat));
	}

	return result.res;
}

int nethostfs_chstat(PspIoDrvFileArg *arg, const char *file, SceIoStat *stat, int bits)
{
	PACKEDSTRUCT
	{
		int cmd;
		IO_CHSTAT_PARAMS params;
	} cmd_params;

	int res;

	strncpy(cmd_params.params.file, file, 256);
	cmd_params.params.fs_num = arg->fs_num;
	memcpy(&cmd_params.params.stat, stat, sizeof(SceIoStat));
	cmd_params.params.bits = bits;

	SEND_COMMAND_WITH_PARAMS(NET_HOSTFS_CMD_IOCHSTAT);
	RETURN_FROM_HOST();
}

int nethostfs_rename(PspIoDrvFileArg *arg, const char *oldname, const char *newname)
{
	PACKEDSTRUCT
	{
		int cmd;
		IO_RENAME_PARAMS params;
	} cmd_params;

	int res;

	strncpy(cmd_params.params.oldfile, oldname, 256);
	strncpy(cmd_params.params.newfile, newname, 256);
	cmd_params.params.fs_num = arg->fs_num;

	SEND_COMMAND_WITH_PARAMS(NET_HOSTFS_CMD_IORENAME);
	RETURN_FROM_HOST();
}

int nethostfs_chdir(PspIoDrvFileArg *arg, const char *dir)
{
	// This function is not called :S
	return 0;
}

int nethostfs_mount(PspIoDrvFileArg *arg)
{
	// Not implemented
	return 0;
}

int nethostfs_umount(PspIoDrvFileArg *arg)
{
	// Not implemented
	return 0;
}

int nethostfs_devctl(PspIoDrvFileArg *arg, const char *devname, unsigned int cmd, void *indata, int inlen, void *outdata, int outlen)
{
	// Not implemented
	return 0;
}
	
int nethostfs_unk21(PspIoDrvFileArg *arg)
{
	// Not implemented
	return 0;
}

PspIoDrvFuncs nethostfs_funcs = 
{
	nethostfs_init,
	nethostfs_exit,
	nethostfs_open,
	nethostfs_close,
	nethostfs_read,
	nethostfs_write,
	nethostfs_lseek,
	nethostfs_ioctl,
	nethostfs_remove,
	nethostfs_mkdir,
	nethostfs_rmdir,
	nethostfs_dopen,
	nethostfs_dclose,
	nethostfs_dread,
	nethostfs_getstat,
	nethostfs_chstat,
	nethostfs_rename,
	nethostfs_chdir,
    nethostfs_mount,
    nethostfs_umount,
    nethostfs_devctl,
    nethostfs_unk21,
};

PspIoDrv nethostfs_driver = 
{
	"nethostfs", 0x10, 0x800, "NetHostFS", &nethostfs_funcs
};

int netHostFSWaitDriverInited()
{
	while (!driverInited)
	{
		sceKernelDelayThread(50000);
	}

	return driverInitError;
}

PspIoDrv *netHostFSGetDriver()
{
	return &nethostfs_driver;
}

int thread_start(SceSize args, void *argp)
{
	char *argv[ARG_MAX+1];
	int argc = 0;
	int i = 0;
	char *p = argp;
	char *drvname;

	while(i < args && argc < ARG_MAX)
    {
		argv[argc] = &p[i];
		i += strlen(&p[i]) + 1;
		argc++;		
	}

    argv[argc] = NULL;
	
	if (argc >= 2)
		strncpy(ip_address, argv[1], 16);
	else
		strcpy(ip_address, "noaddress");	

	drvname = "nethostfs";
	
	if (argc >= 3)
	{
		drvname = argv[2];
		nethostfs_driver.name = argv[2];
	}

	if (pspSdkLoadInetModules() < 0)
	{
		
	}

	if (pspSdkInetInit() < 0)
	{
		sceKernelExitDeleteThread(-1);
		return -1;
	}	

	sceIoDelDrv(drvname);
	driverInitError = sceIoAddDrv(&nethostfs_driver);	
	driverInited = 1;

	sceKernelExitDeleteThread(0);
	return 0;
}

int module_start(SceSize args, void *argp)
{
	SceUID thid = sceKernelCreateThread("nethostfs_start", thread_start, 0x20,
		0x00010000, 0, NULL);	

	if (thid < 0)
	{		
		return -1;
	}

	sceKernelStartThread(thid, args, argp);
	return 0;
}

int module_stop(SceSize args, void *argp)
{
	nethostfs_exit(NULL);

	return 0;
}
