#include <pspsdk.h>
#include <pspkernel.h>
#include <pspdebug.h>
#include <pspctrl.h>
#include <pspumd.h>
#include <pspusb.h>
#include <pspusbstor.h>

#include <stdio.h>
#include <unistd.h>
#include <string.h>

#include "main.h"
#include "umddumper.h"
#include "umdisodumper.h"
#include "umddaxdumper.h"
#include "umdgzipdumper.h"

PSP_MODULE_INFO("UMDDAXDumper", 0x1000, 1, 1);
PSP_MAIN_THREAD_ATTR(0);
PSP_HEAP_SIZE_KB(512);

#define clrscr		pspDebugScreenClear
#define printf		pspDebugScreenPrintf
#define gotoxy		pspDebugScreenSetXY
#define textcolor	pspDebugScreenSetTextColor


char *optionstext[N_OPTIONS] =
{
	"Dump to          : ",
	"Format           : ",
	"Compression level: ",
	"Split size       : "
};

char *destinyoptions[N_DESTINY_OPTIONS] =
{
	"ms0:/ISO       ",
	"usbhostfs0:/ISO",
	"nethostfs0:/ISO"
};

char *destiny_paths[N_DESTINY_OPTIONS] =
{
	"ms0:/ISO/myumd",
	"usbhostfs0:/ISO/myumd",
	"nethostfs0:/ISO/myumd",
};

char *formatoptions[N_FORMAT_OPTIONS] =
{
	"DAX ",
	"ISO ",
	"GZIP"
};

int formats[N_FORMAT_OPTIONS] = 
{
	DUMP_FORMAT_DAX,
	DUMP_FORMAT_ISO,
	DUMP_FORMAT_GZIP
};

char compleveloptions[N_COMPLEVEL_OPTIONS+1] = "123456789";

int complevels[N_COMPLEVEL_OPTIONS] =
{
	1, 2, 3, 4, 5, 6, 7, 8, 9
};

char *splitoptions[N_SPLIT_OPTIONS] =
{
	"No split",
	"400 MB  ",
	"200 MB  ",
	"100 MB  ",
	" 50 MB  ",
	" 20 MB  "
};

int split_sizes[N_SPLIT_OPTIONS] =
{
	0,
	400 * 1048576,
	200 * 1048576,
	100 * 1048576,
	50  * 1048576,
	20  * 1048576,
};


OPTIONS_CHANGE_FUNC optionsChange[N_OPTIONS] =
{
	destinyoptionsChange,
	formatoptionsChange,
	compleveloptionsChange,
	splitoptionsChange
};

char *dump_errors[] =
{
	"",
	"Error reading disc.",
	"Error writing file.",
	"Aborted."
};

int umdSize;

void destinyoptionsChange(int *value, int add)
{
	*value += add;

	if (*value >= N_DESTINY_OPTIONS)
		*value = 0;		
	
	else if (*value < 0)
		*value = N_DESTINY_OPTIONS-1;	

	gotoxy(OPTION_VALUES_X	, DESTINY_OPTIONS_Y);
	printf("%s", destinyoptions[*value]);	
}

void formatoptionsChange(int *value, int add)
{
	*value += add;

	if (*value >= N_FORMAT_OPTIONS)
		*value = 0;		
	
	else if (*value < 0)
		*value = N_FORMAT_OPTIONS-1;

	gotoxy(OPTION_VALUES_X	, FORMAT_OPTIONS_Y);
	printf("%s", formatoptions[*value]);	
}

void compleveloptionsChange(int *value, int add)
{
	*value += add;

	if (*value >= N_COMPLEVEL_OPTIONS)
		*value = 0;		
	
	else if (*value < 0)
		*value = N_COMPLEVEL_OPTIONS-1;

	gotoxy(OPTION_VALUES_X	, COMPLEVEL_OPTIONS_Y);
	printf("%c", compleveloptions[*value]);
}

void splitoptionsChange(int *value, int add)
{
	int indexes[N_SPLIT_OPTIONS];
	int i, n;

	for (i = 0, n = 0; i < N_SPLIT_OPTIONS; i++)
	{
		if (split_sizes[i] < umdSize)
			indexes[n++] = i;
	}

	*value += add;
	
	if (*value >= n)
		*value = 0;

	else if (*value < 0)
		*value = n-1;

	gotoxy(OPTION_VALUES_X	, SPLIT_OPTIONS_Y);
	printf("%s", splitoptions[indexes[*value]]);	
}

void printOption(int index, int value, int highlight)
{
	gotoxy(0, DESTINY_OPTIONS_Y+index);

	textcolor((highlight) ? 0x000000FF : 0x00FFFFFF);
	
	printf("%s", optionstext[index]);
	optionsChange[index](&value, 0);
}

void progress(int value, int total)
{
	gotoxy(PROGRESS_X, 0);
	textcolor(0x00FFFFFF);
	
	printf("Sector %d of %d", value, total);
}

UMDDumper *dumper=NULL;

int dump_thread(SceSize size, void *argp)
{
	int *options = (int *)argp;
	int error;
	char *path = destiny_paths[options[DESTINY_INDEX]];
	int format = formats[options[FORMAT_INDEX]];
	int complevel = complevels[options[COMPLEVEL_INDEX]];
	int splitsize = split_sizes[options[SPLIT_INDEX]];

	switch (format)
	{
		case DUMP_FORMAT_ISO:
			dumper = new UMDIsoDumper(path);
		break;

		case DUMP_FORMAT_DAX:
			dumper = new UMDDaxDumper(path, complevel);
		break;

		case DUMP_FORMAT_GZIP:
			dumper = new UMDGzipDumper(path, complevel);
		break;
	}

	dumper->setPartSize(splitsize);
	dumper->setProgressFunc(progress);

	while ((error = dumper->dumpNextPart()) == DUMP_ERROR_SUCCESS)
	{
		gotoxy(0, NEXT_PART_Y);
		textcolor(0x00FFFFFF);

		printf("Press X to continue with the next part.");

		while (1)
		{
			SceCtrlData pad;

			sceCtrlReadBufferPositive(&pad, 1);

			if (pad.Buttons & PSP_CTRL_CROSS)
				break;

			sceKernelDelayThread(50000);
		}

		gotoxy(0, NEXT_PART_Y);
		printf("                                       ");
	}

	gotoxy(0, DUMP_ERROR_Y);
	textcolor(0x000000FF);

	if (error < 0)
		printf("%s", dump_errors[-error]);
	
	else 
	{
		delete dumper;
		dumper = NULL;
		printf("Dump finished.");
	}

	return 0;
}

void dumpScreen(int *options)
{
	SceUID dumpth;
	char *path;

	clrscr();
	gotoxy(0, 0);
	textcolor(0x00FFFFFF);
	printf("Dumping UMD...");

	gotoxy(0, HELP_Y);
	printf("Press square to abort.");

	path = destiny_paths[options[DESTINY_INDEX]];

	if (strstr(path, "usbhostfs0:/") == path)
	{
		loadUSBHostFS();
	}

	else if (strstr(path, "nethostfs0:/")  == path)
	{
		loadNetHostFS();
	}
	
	dumpth= sceKernelCreateThread("dump_thread", dump_thread, 0x20, 0x10000, 0, NULL);

	if (dumpth >= 0)
		sceKernelStartThread(dumpth, 4*N_OPTIONS, options);

	while (1)
	{
		SceKernelThreadInfo info;
		SceCtrlData pad;	

		sceCtrlReadBufferPositive(&pad, 1);

		if ((pad.Buttons & PSP_CTRL_SQUARE) && dumper)
			dumper->abortDump();		
		
		memset(&info, 0, sizeof(SceKernelThreadInfo));
		info.size = sizeof(SceKernelThreadInfo);
		sceKernelReferThreadStatus(dumpth, &info);

		if (info.status == PSP_THREAD_STOPPED || info.status == PSP_THREAD_KILLED)
			break;		

		sceKernelDelayThread(50000);
	}

	sceKernelWaitThreadEnd(dumpth, NULL);
	sceKernelDeleteThread(dumpth);

	gotoxy(0, HELP_Y);
	printf("If you want to dump another UMD, insert it now and press X.\n");
	printf("If you want to exit, press HOME.\n");

	while (1)
	{
		SceCtrlData pad;
		sceCtrlReadBufferPositive(&pad, 1);

		if (pad.Buttons & PSP_CTRL_CROSS)
			mainScreen();		

		else if (pad.Buttons & PSP_CTRL_HOME)
			exitdumper();

		sceKernelDelayThread(50000);
	}

	sceKernelExitDeleteThread(0);
}

void mainScreen()
{
	int options[N_OPTIONS];
	int i;
	
	mountUMD();
	clrscr();
	textcolor(0x00FFFFFF);

	umdSize = UMDDumper::getUMDSizeBytes();	
	
	printf("UMD Size = %d MB\n", umdSize / 1048576);

	gotoxy(0, HELP_Y);
	printf("Digital up/down: select option\n");
	printf("Digital left/right: change option\n");
	printf("Press start to dump the UMD");
	printf("Press select to load the USB Mass drivers (DO NOT do it if you plan to dump via usb)\n");

	memset(options, 0, sizeof(options));
	options[COMPLEVEL_INDEX] = 9-1;

	for (i = 0; i < N_OPTIONS; i++)
		printOption(i, options[i], (i == 0));

	i = 0;

	while (1)
	{
		SceCtrlData pad;
		int keyprocessed = 0;

		sceCtrlReadBufferPositive(&pad, 1);

		if (pad.Buttons & PSP_CTRL_UP)
		{
			if (i != 0)
			{
				printOption(i, options[i], 0);
				printOption(i-1, options[i-1], 1);
				i--;

				keyprocessed = 1;
			}			
		}
		else if (pad.Buttons & PSP_CTRL_DOWN)
		{
			if (i != (N_OPTIONS-1))
			{
				printOption(i, options[i], 0);
				printOption(i+1, options[i+1], 1);
				i++;

				keyprocessed = 1;
			}
		}
		else if (pad.Buttons & PSP_CTRL_LEFT)
		{
			// Avoid changing the split size when format is GZIP
			if (i == SPLIT_INDEX && 
				formats[options[FORMAT_INDEX]] == DUMP_FORMAT_GZIP);
					
			else 
			{
				textcolor(0x000000FF);
				optionsChange[i](&options[i], -1);

				if (i == FORMAT_INDEX && formats[options[FORMAT_INDEX]] == DUMP_FORMAT_GZIP)
				{
					textcolor(0x00FFFFFF);
					options[SPLIT_INDEX] = 0;
					optionsChange[SPLIT_INDEX](&options[SPLIT_INDEX], 0);
				}
			
				keyprocessed = 1;
			}
		}
		else if (pad.Buttons & PSP_CTRL_RIGHT)
		{
			// Avoid changing the split size when format is GZIP
			if (i == SPLIT_INDEX && 
				formats[options[FORMAT_INDEX]] == DUMP_FORMAT_GZIP);
					
			else 
			{
				textcolor(0x000000FF);
				optionsChange[i](&options[i], +1);

				if (i == FORMAT_INDEX && formats[options[FORMAT_INDEX]] == DUMP_FORMAT_GZIP)
				{
					textcolor(0x00FFFFFF);
					options[SPLIT_INDEX] = 0;
					optionsChange[SPLIT_INDEX](&options[SPLIT_INDEX], 0);
				}
			
				keyprocessed = 1;
			}
		}
		else if (pad.Buttons & PSP_CTRL_START)
		{
			dumpScreen(options);
		}
		else if (pad.Buttons & PSP_CTRL_SELECT)
		{
			loadUSBMass();
		}

		else if (pad.Buttons & PSP_CTRL_HOME)
			exitdumper();	
		
		sceKernelDelayThread((keyprocessed) ? 200000 : 50000);
	}
}

void loadUSBMass()
{
	if (!sceKernelFindModuleByName("USBHostFS"))
	{
		pspSdkLoadStartModule("flash0:/kd/semawm.prx", PSP_MEMORY_PARTITION_KERNEL);
		pspSdkLoadStartModule("flash0:/kd/usbstor.prx", PSP_MEMORY_PARTITION_KERNEL);
		pspSdkLoadStartModule("flash0:/kd/usbstormgr.prx", PSP_MEMORY_PARTITION_KERNEL);
		pspSdkLoadStartModule("flash0:/kd/usbstorms.prx", PSP_MEMORY_PARTITION_KERNEL);
		pspSdkLoadStartModule("flash0:/kd/usbstorboot.prx", PSP_MEMORY_PARTITION_KERNEL);
		sceUsbStart(PSP_USBBUS_DRIVERNAME, 0, 0);
		sceUsbStart(PSP_USBSTOR_DRIVERNAME, 0, 0);
		sceUsbstorBootSetCapacity(0x800000);
		sceUsbActivate(0x1c8);
	}
}

void loadUSBHostFS()
{
	char mod[256];

	getcwd(mod, 256);
	strcat(mod, "/usbhostfs.prx");

	if (pspSdkLoadStartModule(mod, PSP_MEMORY_PARTITION_KERNEL) >= 0)
	{
		sceUsbStart(PSP_USBBUS_DRIVERNAME, 0, 0);
		sceUsbStart("USBHostFSDriver", 0, 0);
		sceUsbActivate(0x1C9);
		
		// Wait a bit for driver initialization 
		sceKernelDelayThread(4 * 1000 * 1000);
		sceIoAssign("usbhostfs0:", "host0:", NULL, IOASSIGN_RDWR, NULL, 0);
	}	
}

extern "C" int netHostFSWaitDriverInited();

void loadNetHostFS()
{
	char cwd[256];
	char file[256];
	char ip[16];

	getcwd(cwd, 256);

	sprintf(file, "%s/%s", cwd, "ip.txt");
	
	SceUID ipfile = sceIoOpen(file, PSP_O_RDONLY, 0777);

	if (ipfile >= 0)
	{
		memset(ip, 0, 16);
		sceIoRead(ipfile, ip, 16);
		sceIoClose(ipfile);

		sprintf(file, "%s/%s", cwd, "nethostfs.prx");		

		char *p = ip;

		pspSdkLoadStartModuleWithArgs(file, PSP_MEMORY_PARTITION_KERNEL, 1, &p);
		netHostFSWaitDriverInited();		
	}
}

void mountUMD()
{
	if (!sceUmdCheckMedium(0))
	{
		printf("Insert an UMD.\n");
		sceUmdWaitDriveStat(UMD_WAITFORDISC);
	}

	sceUmdActivate(1, "disc0:");
	sceUmdWaitDriveStat(UMD_WAITFORINIT);
}

extern "C" u32 sceKernelSearchModuleByName (const char *);

void exitdumper()
{
	// stopUSB();
	u32 mod = sceKernelSearchModuleByName("NetHostFS");

	if (mod >= 0)
	{
		sceKernelStopModule(mod, 0, NULL, NULL, NULL);
	}

	sceKernelExitGame();
}

/* Pause the core of umdemulator/daxziso */
void pauseCore(/*CORE_CONTEXT *context*/)
{
	/*if (context)
	{
		context->ioopen_inst1 = _lw(0x8804d80c);
		context->ioopen_inst2 = _lw(0x8804d810);

		context->ioopenasync_inst1 = _lw(0x8804d828);
		context->ioopenasync_inst2 = _lw(0x8804d82c);

		context->iodevctl_inst1 = _lw(0x8804b944);
		context->iodevctl_inst2 = _lw(0x8804b948);
	}*/

	// Restore sceIoOpen
	_sw(0x27bdfff0, 0x8804d80c);
	_sw(0xafbf0000, 0x8804d810);

	// Restore sceIoOpenAsync
	_sw(0x27bdfff0, 0x8804d828);
	_sw(0xafbf0000, 0x8804d82c);

	// Restore sceIoDevctl
	_sw(0x27bdffc0, 0x8804b944);
	_sw(0xafbe0030, 0x8804b948);

	sceKernelDcacheWritebackAll();
}


int main(int argc, char *argv[])
{
	pspDebugScreenInit();
	pspSdkInstallNoDeviceCheckPatch();
	pspSdkInstallNoPlainModuleCheckPatch();

	/**(strrchr(argv[0], '/')) = 0;
	sceIoChdir(argv[0]);
	printf("argv[0]: %s\n", argv[0]);
	sceKernelDelayThread(50 * 1000 * 1000);*/
	
	pauseCore();

	sceCtrlSetSamplingCycle(0);
	sceCtrlSetSamplingMode(PSP_CTRL_MODE_DIGITAL);

	mainScreen();

	while (1)
	{
		sceKernelDelayThread(50000);
	}
	
	return 0;
}
