/*
 * NetHostFS by Dark_AleX
 *
 * Inspired by psplink
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>

#include "nethostfs.h"

#define MAX_OPENED_FILES		1024
#define MAX_OPENED_DIRS			 256
#define	MAX_PATH				1024	

typedef struct FileData
{
	char	name[256];
	int		fd;
	int		mode;			
} FileData;

typedef struct DirData
{
	char		name[256];
	SceIoDirent *entries;
	int			pos;
	int			nentries;
	int			opened;
} DirData;

char	rootdir[512];

FileData	files_table[MAX_OPENED_FILES];
DirData		dirs_table[MAX_OPENED_DIRS];

#define SEND_INTEGER(x) if (send(sock, &x, 4, 0) != 4) return -1
#define RECV_INTEGER(x) if (recv(sock, x, 4, 0) != 4) return -1
#define RECV_PARAMS() if (recv(sock, &params, sizeof(params), 0) != sizeof(params)) return -1
#define SEND_RETURN(x) res = x; if (send(sock, &res, 4, 0) != 4) return -1
#define SEND_RESULT() if (send(sock, &result, sizeof(result), 0) != sizeof(result)) return -1
#define MAKE_PATH(s) snprintf(path, MAX_PATH, "%s%s", rootdir, s)

/* Converts native time to psp time */
void psptime(time_t *t, ScePspDateTime *pspt)
{
	struct tm *filetime;

	memset(pspt, 0, sizeof(ScePspDateTime));
	filetime = localtime(t);
	pspt->year = filetime->tm_year + 1900;
	pspt->month = filetime->tm_mon + 1;
	pspt->day = filetime->tm_mday;
	pspt->hour = filetime->tm_hour;
	pspt->minute = filetime->tm_min;
	pspt->second = filetime->tm_sec;
}

/* Converts psp time to native time */
void nativetime(ScePspDateTime *pspt, time_t *t)
{
	struct tm stime;

	stime.tm_year = pspt->year - 1900;
	stime.tm_mon = pspt->month - 1;
	stime.tm_mday = pspt->day;
	stime.tm_hour = pspt->hour;
	stime.tm_min = pspt->minute;
	stime.tm_sec = pspt->second;

	*t = mktime(&stime);
}

/* Converts native stat to psp stat */
void pspstat(struct stat *st, SceIoStat *pspst)
{
	pspst->st_size = st->st_size;
	pspst->st_mode = 0;
	pspst->st_attr = 0;
	
	if(S_ISLNK(pspst->st_mode))
	{
		pspst->st_attr = FIO_SO_IFLNK;
		pspst->st_mode = FIO_S_IFLNK;
	}
	else if(S_ISDIR(st->st_mode))
	{
		pspst->st_attr = FIO_SO_IFDIR;
		pspst->st_mode = FIO_S_IFDIR;
	}
	else
	{
		pspst->st_attr = FIO_SO_IFREG;
		pspst->st_mode = FIO_S_IFREG;
	}

	pspst->st_mode |= (st->st_mode & (S_IRWXU | S_IRWXG | S_IRWXO));
	psptime(&st->st_ctime, &pspst->stctime);
	psptime(&st->st_atime, &pspst->statime);
	psptime(&st->st_mtime, &pspst->stmtime);
}

int process_hello(int sock)
{
	int res;

	SEND_RETURN(NET_HOSTFS_CMD_HELLO);

	return 0;
}

int process_ioinit(int sock)
{
	int i, res;

	memset(files_table, 0, sizeof(files_table));
	memset(dirs_table, 0, sizeof(dirs_table));

	for (i = 0; i < MAX_OPENED_FILES; i++)
		files_table[i].fd = -1;
	
	getcwd(rootdir, 512);	

	SEND_RETURN(0);

	return 0;
}

int process_ioexit(int sock)
{
	return -1; // End
}

int process_ioopen(int sock)
{
	IO_OPEN_PARAMS params;
	char path[MAX_PATH];
	int i, flags, res;
	
	RECV_PARAMS();

	for (i = 0; i < MAX_OPENED_FILES; i++)
	{
		if (files_table[i].fd < 0)
			break;
	}

	if (i == MAX_OPENED_FILES)
	{
		SEND_RETURN(-1);
		return 0;
	}

	strncpy(files_table[i].name, params.file, 256);	
	MAKE_PATH(files_table[i].name);

	flags = 0;

	if (params.flags & PSP_O_RDONLY)
		flags |= O_RDONLY;

	if (params.flags & PSP_O_WRONLY)
		flags |= O_WRONLY;

	if (params.flags & PSP_O_APPEND)
		flags |= O_APPEND;

	if (params.flags & PSP_O_CREAT)
		flags |= O_CREAT;

	if (params.flags & PSP_O_TRUNC)
		flags |= O_TRUNC;

	if (params.flags & PSP_O_EXCL)
		flags |= O_EXCL;	

	files_table[i].fd = open(path, flags, params.mode);

	if (files_table[i].fd < 0)
	{
		SEND_RETURN(files_table[i].fd);	
	}	
	else 
	{ 
		SEND_RETURN(i); 
	}

	return 0;
}

int process_ioclose(int sock)
{
	int fd, res;

	RECV_INTEGER(&fd);

	if (fd < 0 || fd >= MAX_OPENED_FILES)
	{
		SEND_RETURN(-1);
	}	
	else if (close(files_table[fd].fd) < 0)
	{
		SEND_RETURN(-1);
	}		
	else
	{
		files_table[fd].fd = -1;
		SEND_RETURN(0);
	}

	return 0;
}

int process_ioread(int sock)
{
	IO_READ_PARAMS params;
	int bytesread, res;
	char *buf;

	RECV_PARAMS();

	// Limit of 32 MB (which anyways it has no sense in the psp)
	if (params.len >= (32 * 1048576))
	{
		SEND_RETURN(-1);
		return 0;
	}
	else if (params.len <= 0)
	{
		SEND_RETURN(0);
		return 0;
	}
	else if (params.fd < 0 || params.fd >= MAX_OPENED_FILES)
	{
		SEND_RETURN(-1);
		return 0;
	}
	else if (files_table[params.fd].fd < 0)
	{
		SEND_RETURN(-1);
		return 0;
	}	

	buf = (char *)malloc(params.len);
	if (!buf)
	{
		SEND_RETURN(-1);
		return 0;
	}

	bytesread = read(files_table[params.fd].fd, buf, params.len);
	SEND_RETURN(bytesread);

	if (bytesread > 0)
	{
		int x = 0, sent = 0;
		char *pbuf = buf;

		while (sent < bytesread)
		{
			int blocksize;

			if ((bytesread - sent) >= 2048)
				blocksize = 2048;
			else
				blocksize = bytesread - sent;

			x = send(sock, pbuf, blocksize, 0);
			
			if (x <= 0)
				return -1;

			sent += x;
			pbuf += x;
		}
	}

	free(buf);
	return 0;
}

int process_iowrite(int sock)
{
	IO_WRITE_PARAMS params;
	int error, x, received, res;
	char *buf, *pbuf;

	RECV_PARAMS();
	
	if (params.len >= (32 * 1048576))
	{
		error = 1;
	}
	else if (params.len <= 0)
	{
		SEND_RETURN(0);
		return 0;
	}
	else if (params.fd < 0 || params.fd >= MAX_OPENED_FILES)
	{
		error = 1;
	}
	else if (files_table[params.fd].fd < 0)
	{
		error = 1;
	}

	x = 0;
	received = 0;
	
	if (error)
	{
		buf = (char *)malloc(2048);
	}
	else
	{
		buf = (char *)malloc(params.len);
	}

	if (!buf)
		return -1;
			
	pbuf = buf;

	while (received < params.len)
	{
		int blocksize;

		if ((params.len - received) >= 2048)
			blocksize = 2048;
		else
			blocksize = params.len - received;
		
		x = recv(sock, pbuf, blocksize, 0);
		
		if (x <= 0)
			return -1;

		received += x;

		if (!error)
			pbuf += x;
	}

	if (error)
	{
		SEND_RETURN(-1);
	}

	else
	{
		SEND_RETURN(write(files_table[params.fd].fd, buf, params.len));
	}

	free(buf);
	return 0;
}

int process_iolseek(int sock)
{
	IO_LSEEK_PARAMS params;
	int res;

	RECV_PARAMS();

	if (params.fd < 0 || params.fd >= MAX_OPENED_FILES || files_table[params.fd].fd < 0)
	{
		SEND_RETURN(-1);
	}
	else
	{
		SEND_RETURN(lseek(files_table[params.fd].fd, params.offset, params.whence));
	}

	return 0;
}

int process_ioremove(int sock)
{
	IO_REMOVE_PARAMS params;
	char path[MAX_PATH];
	int res;

	RECV_PARAMS();
	params.file[255] = 0;
	MAKE_PATH(params.file);
	
	SEND_RETURN(remove(path));
	return 0;
}

int process_iomkdir(int sock)
{
	IO_MKDIR_PARAMS params;
	char path[MAX_PATH];
	int res;

	RECV_PARAMS();
	params.dir[255] = 0;
	MAKE_PATH(params.dir);
	
	SEND_RETURN(mkdir(path, params.mode));
	return 0;
}

int process_iormdir(int sock)
{
	IO_RMDIR_PARAMS params;
	char path[MAX_PATH];
	int res;

	RECV_PARAMS();
	params.dir[255] = 0;
	MAKE_PATH(params.dir);
	
	SEND_RETURN(rmdir(path));
	return 0;
}

int process_iodopen(int sock)
{
	IO_DOPEN_PARAMS params;
	char path[MAX_PATH];
	struct dirent **entries;
	int i, j, n, res;

	RECV_PARAMS();
	
	for (i = 0; i < MAX_OPENED_DIRS; i++)
	{
		if (dirs_table[i].entries == NULL)
			break;
	}

	if (i == MAX_OPENED_DIRS)
	{
		SEND_RETURN(-1);
		return 0;
	}

	MAKE_PATH(params.dir);
	n = scandir(path, &entries, NULL, alphasort);

	if (n < 0)
	{
		SEND_RETURN(n);
	}
	else
	{
		dirs_table[i].opened = 1;

		if (n != 0)
		{
			dirs_table[i].entries = (SceIoDirent *)malloc(n * sizeof(SceIoDirent));

			if (!dirs_table[i].entries)
			{
				SEND_RETURN(-1);
				return 0;
			}

			dirs_table[i].pos = 0;
			dirs_table[i].nentries = n;

			for (j = 0; j < n; j++)
			{
				char filepath[MAX_PATH];
				struct stat st;

				snprintf(filepath, MAX_PATH, "%s/%s", path, entries[j]->d_name);

				memset(&dirs_table[i].entries[j], 0, sizeof(SceIoDirent));
				strncpy(dirs_table[i].entries[j].d_name, entries[j]->d_name, 256);
				
				if (stat(filepath, &st) < 0)
				{
					free(dirs_table[i].entries);
					dirs_table[i].entries = NULL;
					
					SEND_RETURN(-1);
					return 0;
				}

				pspstat(&st, &dirs_table[i].entries[j].d_stat);
			}
		}		

		SEND_RETURN(i);
	}

	return 0;
}

int process_iodclose(int sock)
{
	int fd, res;

	RECV_INTEGER(&fd);

	if (fd < 0 || fd >= MAX_OPENED_DIRS)
	{
		SEND_RETURN(-1);
	}
	else if (!dirs_table[fd].opened)
	{
		SEND_RETURN(-1);
	}
	else if (!dirs_table[fd].entries)
	{
		dirs_table[fd].opened = 0;
		SEND_RETURN(0);
	}
	else
	{
		free(dirs_table[fd].entries);
		dirs_table[fd].entries = NULL;
		dirs_table[fd].opened = 0;
		SEND_RETURN(0);
	}	
	
	return 0;
}

int process_iodread(int sock)
{
	IO_DREAD_RESULT result;
	int fd;

	RECV_INTEGER(&fd);

	memset(&result, 0, sizeof(result));

	if (fd < 0 || fd >= MAX_OPENED_DIRS)
	{
		result.res = -1;
	}
	else if (!dirs_table[fd].opened)
	{
		result.res = -1;
	}
	else if (!dirs_table[fd].entries)
	{
		result.res = 0;
	}
	else
	{
		int pos = dirs_table[fd].pos;
		
		if (pos >= dirs_table[fd].nentries)
		{
			result.res = 0; // No more entries
		}
		else
		{
			result.res = pos+1;
			memcpy(&result.entry, &dirs_table[fd].entries[pos], sizeof(SceIoDirent));
			dirs_table[fd].pos++;
		}
	}

	SEND_RESULT();
	return 0;
}

int	process_iogetstat(int sock)
{
	IO_GETSTAT_PARAMS params;
	IO_GETSTAT_RESULT result;
	char path[MAX_PATH];
	struct stat st;

	RECV_PARAMS();
	MAKE_PATH(params.file);

	memset(&result, 0, sizeof(result));
	memset(&st, 0, sizeof(st));

	if (stat(path, &st) < 0)
	{
		result.res = -1;
	}
	else
	{
		result.res = 0;
		pspstat(&st, &result.stat);		
	}

	SEND_RESULT();
	return 0;
}

int process_iochstat(int sock)
{
	IO_CHSTAT_PARAMS params;
	char path[MAX_PATH];
	int res;

	RECV_PARAMS();
	MAKE_PATH(params.file);

	if (params.bits & PSP_CHSTAT_MODE)
	{
		if (chmod(path, params.stat.st_mode & (FIO_S_IRWXU | FIO_S_IRWXG | FIO_S_IRWXO)) < 0)
			SEND_RETURN(-1);
	}

	SEND_RETURN(0);
	return 0;
}

int process_iorename(int sock)
{
	IO_RENAME_PARAMS params;
	char oldpath[MAX_PATH], newpath[MAX_PATH];
	int res;

	RECV_PARAMS();
	
	sprintf(oldpath, "%s%s", rootdir, params.oldfile);
	sprintf(newpath, "%s%s", rootdir, params.newfile);

	SEND_RETURN(rename(oldpath, newpath));
	return 0;
}

int process_cmd(int sock)
{
	int cmd, res;
	int (* process_func)(int);

	RECV_INTEGER(&cmd);	

	process_func = NULL;
	//printf("cmd = %08X\n", cmd);

	switch (cmd)
	{
		case NET_HOSTFS_CMD_HELLO:	
			process_func = process_hello;
		break;

		case NET_HOSTFS_CMD_IOINIT:
			process_func = process_ioinit;	
		break;
		
		case NET_HOSTFS_CMD_IOEXIT:
			process_func = process_ioexit;
		break;

		case NET_HOSTFS_CMD_IOOPEN:
			process_func = process_ioopen;
		break;

		case NET_HOSTFS_CMD_IOCLOSE:
			process_func = process_ioclose;
		break;

		case NET_HOSTFS_CMD_IOREAD:
			process_func = process_ioread;
		break;

		case NET_HOSTFS_CMD_IOWRITE:
			process_func = process_iowrite;
		break;

		case NET_HOSTFS_CMD_IOLSEEK:
			process_func = process_iolseek;
		break;

		case NET_HOSTFS_CMD_IOREMOVE:
			process_func = process_ioremove;
		break;

		case NET_HOSTFS_CMD_IOMKDIR:
			process_func = process_iomkdir;
		break;

		case NET_HOSTFS_CMD_IORMDIR:
			process_func = process_iormdir;
		break;

		case NET_HOSTFS_CMD_IODOPEN:
			process_func = process_iodopen;
		break;

		case NET_HOSTFS_CMD_IODCLOSE:
			process_func = process_iodclose;
		break;

		case NET_HOSTFS_CMD_IODREAD:
			process_func = process_iodread;
		break;

		case NET_HOSTFS_CMD_IOGETSTAT:
			process_func = process_iogetstat;
		break;

		case NET_HOSTFS_CMD_IOCHSTAT:
			process_func = process_iochstat;
		break;

		case NET_HOSTFS_CMD_IORENAME:
			process_func = process_iorename;
		break;

		default:
			printf("Unknown command: 0x%08X\n", cmd);
	}

	if (process_func)
		res = process_func(sock);
	else
		res = 0;

	if (res >= 0)
	{
		if (cmd >= 0)
			res = cmd;
		else
			res = 0;
	}

	return res;
}

int main()
{
	int sock;
	struct sockaddr_in addr, claddr;	

	sock = socket(PF_INET, SOCK_STREAM, 0);

	if (sock < 0)
	{
		printf("Error in socket.\n");
		return -1;
	}

	addr.sin_family = AF_INET;
	addr.sin_port = htons(7513);
	addr.sin_addr.s_addr = INADDR_ANY;
	
	if (bind(sock, (struct sockaddr *)&addr, sizeof(addr)) < 0)
	{
		printf("Error in bind\n");
		return -1;
	}

	if (listen(sock, 1) < 0)
	{
		printf("Error in listen\n");
		return -1;
	}

	while (1)
	{
		int consock, flag, size;

		size = sizeof(claddr);
		consock = accept(sock, (struct sockaddr *)&claddr, &size);
		if (consock < 0)
		{
			printf("Error in accept\n");
			return 0;
		}

		flag = 1;
		setsockopt(consock, SOL_TCP, TCP_NODELAY, &flag, sizeof(flag));

		printf("Accepted connection from IP %s\n", inet_ntoa(claddr.sin_addr));

		if (process_cmd(consock) == NET_HOSTFS_CMD_HELLO)
		{
			while (process_cmd(consock) >= 0);
		}
		
		close(consock);
		printf("Connection closed.\n");
	}


	return 0;
}
