/*
 * umedax.h (Dark_AleX)
 *
 * This file contains defintions of constants, structures and 
 * functions to handle UME/DAX config files, and also to 
 * install/uninstall the UME or UME+DAX cores to/from memory.
*/

#ifndef _UMEDAX_H_
#define _UMEDAX_H_

#define UME_CONFIGFILE_SIGNATURE	0x0000033e


/* Constants for UMEConfig/UMEDAXConfig */

enum UME_Speed
{
	UME_SPEED_222 = 0,
	UME_SPEED_266 = 1,
	UME_SPEED_333 = 2,
};

enum UME_EmuMode
{
	UME_EMUMODE_OFF = 0,
	UME_EMUMODE_SYSTEMMENU = 1,
	UME_EMUMODE_DIRECTLOAD = 2,
	UME_EMUMODE_COPYLOAD = 3,
};

enum UME_UpdateIcon
{
	UME_UPDATEICON_ORIGINAL = 0,
	UME_UPDATEICON_MEMSTICK = 1,
	UME_UPDATEICON_REMOVE = 2,
};

enum PSP_DiscType
{
	PSP_DISCTYPE_GAME  = 0x00000010,
	PSP_DISCTYPE_VIDEO = 0x00000020,
	PSP_DISCTYPE_AUDIO = 0x00000040,
};

enum UME_DiscType
{
	UME_DISCTYPE_GAME  = 0,
	UME_DISCTYPE_VIDEO = 1,
	UME_DISCTYPE_AUDIO = 2,
}; 

enum PSP_Region
{
	PSP_REGION_JAPAN = 0x00000001,
	PSP_REGION_AMERICA = 0x00000002,
	PSP_REGION_EUROPE = 0x00000004,
	PSP_REGION_FREE = 0x0000000f,
}; 

enum UME_Region
{
	UME_REGION_JAPAN = 0,
	UME_REGION_AMERICA = 1,
	UME_REGION_EUROPE = 2,
	UME_REGION_FREE = 3,
};

enum DAX_EmuMode
{
	DAX_EMUMODE_OFF = UME_EMUMODE_OFF,
	DAX_EMUMODE_SYSTEMMENU = UME_EMUMODE_SYSTEMMENU,
	DAX_EMUMODE_DIRECTLOAD = UME_EMUMODE_DIRECTLOAD,
	DAX_EMUMODE_COPYLOAD = UME_EMUMODE_COPYLOAD,
	DAX_EMUMODE_MPHGL = 4,
	DAX_EMUMODE_ORIGINALUME = 5,
	DAX_EMUMODE_SYSTEMMENUMPHGL = 6,
	DAX_EMUMODE_VSHEXECMPHGL = 7,
	DAX_EMUMODE_DIRECTLOADEXEC = 8,
	DAX_EMUMODE_MPHGLOADEXEC = 9,
	DAX_EMUMODE_RUNUMDLOADEXEC = 10,
	DAX_EMUMODE_SYSTEMMENURUNUMD = 11,
};


/* Pointers */

#define UME_CORE_ADDRESS			0x883e0000
#define UMEDAX_CORE_ADDRESS			UME_CORE_ADDRESS
#define UME_CORECONFIG_ADDRESS		0x883e0f80
#define UME_ISOSTRING_ADDRESS		0x883e0fc0

#define BOOTSTRAP_INST1				0x88069684
#define BOOTSTRAP_INST2				0x880bce84

#define UME_INITCORE_ADDRESS		0x883e009c
#define DAX_INITCORE_ADDRESS		0x883e1678


/* Definitions of strcutures */

typedef struct
{
	/* One of UME_Speed values */
	u32 speed;
	/* Boolean value indicating whether emulate flash0 or not */
	u32 f0emu;
	/* Boolean value indicating whether emulate flash1 or not */
	u32 f1emu;
	/* One of UME_EmuMode values 
	 * The UME core only checks if this is zero(UME_EMUMODE_OFF)
	 * or not zero to know ifit should emulate the UMD.
	 * The rest of possible values are handled by the UME executable.
	 * The DAX executable use the DAXCoreConfig.emulationmode instead
	 * and it tries to keep both variables with the same value.
	 * See DAXCoreConfig for more info  */
	u32 emulationmode; 
	/* One of UME_UpdateIcon values */	
	u32 updateicon;
	/* One of PSP_DiscType values. This refers to the emulated UMD. 
	 * To convert a value from UME_DiscType to PSP_DiscType use the
	 * UME_DiscType2PSP_DiscType function */
	u32 disctypeISO;
	/* The same as above, but it refers to the real UMD */
	u32 disctypeUMD;
	/* Boolean value indicating whether use the BOOT.BIN or not.
	   It seems (i've not confirmed it) that this value is not used
	   by the UME core despite being here, that means that the 
	   executable is the only one responsable of handling this value */
	u32 usebootbin;
	/* Boolean value indicating whether override the umd region code
	   or not */
	u32 regcodeoverride;
	/* One of PSP_Region values. This for game ISO's/UMD's 
	 * To convert a value from UME_Region to PSP_Region, use the
	 * UME_Region2PSP_Region function */
	u32 regcodegame;
	/* The same as above but for video ISO's/UMD's */
	u32 regcodevideo;
	/* The same as above but fot audio ISO's/UMD's */
	u32 regcodeaudio;
} UMECoreConfig;

typedef struct
{
	/* One of DAX_EmuMode values 
	 * The DAX core only checks if this is one of MPHGL types
	 * to know if it should redirect the call that the firmware
	 * does when executing a GAME in system menu from 
	 * sceKernelLoadExecVSHDisc("disc0:/PSP_GAME/SYSDIR/EBOOT.BIN", arg2)
	 * to LoadExecForKernel_28D0D249("ms0:/MPHGameLoader/EBOOT.PBP", arg2)
	 * If the value is DAX_EMUMODE_VSHEXECMPHGL, the DAX core will
	 * also restart (only the first time the core is executed) the VSH 
	 * with the sceKernelExitVSHVSH function.
	 * Almost all values, including some of the MPHGL values are
	 * handled by the DAX executable.
	 * The DAX core doesn't need to check if the value is 
	 * UME_EMUMODE_OFF, because that is already done by the UME core.
	*/
	u32 emulationmode;
	/* Boolean value indicating whether allocate tables in RAM 
	   or not */
	u32 allocatetables;
} DAXCoreConfig;

/* This is the 72 bytes structure used by the UME executable to save
 * the configuration to ms0:/UMDEMULATOR/config.dat */  
typedef struct
{
	/* The signature. It must be UME_CONFIGFILE_SIGNATURE */
	u32 signature;
	/* The configuration of the UME core (see UMECoreConfig) */
	UMECoreConfig coreconfig;
	/* One of UME_DiscType values. This refers to the emulated UMD. 
	 * To convert a value from UME_DiscType to PSP_DiscType use the
	 * UME_DiscType2PSP_DiscType function */
	u32 disctypeISO; 
	/* The same as above but it refers to the real UMD */
	u32 disctypeUMD; 
	/* One of UME_Region values. This for game ISO's/UMD's 
	 * To convert a value from UME_Region to PSP_Region, use the
	 * UME_Region2PSP_Region function */
	u32 regiongame; 
	/* The same as above but for video ISO's/UMD's */
	u32 regionvideo; 
	/* The same as above but for audio ISO's/UMD's */
	u32 regionaudio; 
} UMEConfig;

/*  This is the (currently) 80 bytes structure used by the DAX
 *  executable to save the configuration of each ISO. 
 *  It's the same one used by the UME executable, except that it
 *  has the DAXCoreConfig appended to the end. */
typedef struct
{
	u32 signature;
	UMECoreConfig coreconfig;
	u32 disctypeISO; 
	u32 disctypeUMD; 
	u32 regiongame; 
	u32 regionvideo; 
	u32 regionaudio; 
	/* The configuration of the DAX core. (See DAXCoreConfig) */
	DAXCoreConfig daxcoreconfig;
} UMEDAXConfig;


/* Types for the umeCoreInit and daxCoreInit functions */
typedef void (* UME_INIT_TYPE) (void);
typedef void (* DAX_INIT_TYPE) (DAXCoreConfig *);


/* Prototypes of functions */


/*
 * It converts the given UME_DiscType value to
 * a PSP_DiscType value.
 * The value must be a valid UME_DiscType value, otherwise an 
 * exception could ocurr.
 *
 * @param value (IN): the value to convert
 *
 * @return: the value converted to PSP_DiscType.
*/
u32 UME_DiscType2PSP_DiscType(u32 value);

/*
 * It converts the given UME_Region value to
 * a PSP_Region value.
 * The value must be a valid UME_Region, otherwise an exception 
 * could ocurr.
 *
 * @param value (IN): the value to convert
 *
 * @return: the value converted to PSP_Region.
*/
u32 UME_Region2PSP_Region(u32 value);


/*
 * It fills the given structure with the default UMEDAX configuration.
 * 
 * @param config (OUT): a pointer to the structure to be filled
 * with the default UMEDAX configuration.
*/
void defaultConfig(UMEDAXConfig *config);

/*
 * It fills the given structure with the UMEDAX configuration found
 * in the file specified by the param "file".
 * If the file cannot be opened or the signature of the file is not
 * equal to UME_CONFIGFILE_SIGNATURE, then it fills the structure
 * with the default UMEDAX configuration.
 * If the files can be opened and the signature of the file is 
 * UME_CONFIGFILE_SIGNATURE, but the size of the file (= X bytes)
 * is less than sizeof(UMEDAXConfig), then it fills the first X
 * bytes of the structure with the UMEDAX configuration found in the 
 * file, and the rest with the default comfiguration
 * If the file is greater than sizeof(UMEDAXConfig), only the first
 * sizeof(UMEDAXConfig) bytes are read and the rest of the file is
 * ignored.
 * The function also corrects some invalid values that can be in
 * the input file to avoid the caller worry about values out of bounds,
 * etc.
 *
 * @param file (IN): the path of the file containing the UMEDAX 
 * configuration.
 * @param config (OUT): pointer to the structure to be filled with
 * the UMEDAX configuration found in the file pointed by the parameter
 * "file".
*/
void readConfig(char *file, UMEDAXConfig *config);

/* It writes the UMEDAX configuration pointed by the "config"
 * parameter to the file pointed  by the "file" parameter.
 * If the file exists, the file is overwritten.
 * If the file doesn't exist, a new one is created, but the path
 * of the file must be accesible.
 * The function doesn't check if the values are correct or not since
 * the readConfig function already does that and it's the obligation
 * of the application to keep those values correct.
 *
 * @param file (IN): the path of the file to be created/written
 * with the UMEDAX configuration 
 * @param config (IN): pointer to the structure containing the
 * UMEDAX configuration to be written to the file pointed by the
 * parameter "file".
*/
void writeConfig(char *file, UMEDAXConfig *config);

/*
 * It installs in Kernel RAM (0x883e0000) the UME core from
 * the plain mips binary file pointed by the parameter "core".
 * The core will remain resident in the system surviving again
 * and again to the psp restart that it's produced after each
 * call to sceKernelExitGame, sceKernelLoadExec, sceKernelExitVSHVSH,
 * sceKernelLoadExecVSHDisc, LoadExecForKernel_28D0D249 and all other
 * functions that restart the psp, until a call to unistallCore is
 * done.
 * In order to success in emulating the UMD, the ISO file pointed
 * by the parameter "iso" has to be a valid ISO9660 file.
 * If config->emulationmode is equal to UME_EMUMODE_OFF, the UMD
 * won't be emulated.
 *
 * @param core (IN): path to the plain mips binary file containing
 * the UME core to be installed in the system.
 * @param iso (IN): path to the ISO9660 file that will be used to 
 * emulate the UMD.
 * @param config (IN): pointer to the structure containing the 
 * configuration of the UME core.
 *
 * @return: -1 if the core cannot be installed (the file pointed
 * by "core" doesn't exist) or 0 if success.
 *
 * @Remarks:
 * If the file pointed by the parameter "core" is not a valid
 * UME core, the function won't return and the psp will crash
 * if an exception handler is not installed.
 *
 * @Example of usage:
 * installUMECore("ms0:/UMDEMULATOR/core.bin", "ms0:/ISO/mygame.iso", &myUMEcore_config);
 */
int installUMECore(char *core, char *iso, UMECoreConfig *config);

/*
 * It installs in Kernel RAM (0x883e0000) the UME+DAX core from
 * the plain mips binary file pointed by the parameter "core".
 * The core will remain resident in the system surviving again
 * and again to the psp restart that it's produced after each
 * call to sceKernelExitGame, sceKernelLoadExec, sceKernelExitVSHVSH,
 * sceKernelLoadExecVSHDisc, LoadExecForKernel_28D0D249 and all other
 * functions that restart the psp, until a call to unistallCore is
 * done.
 * In order to success in emulating the UMD, the ISO or DAX file 
 * pointed by the parameter "iso" has to be a valid ISO9660 file or 
 * a valid DAX file produced from an ISO9660 file.
 * If config->emulationmode is equal to UME_EMUMODE_OFF, the UMD
 * won't be emulated.
 *
 * @param core (IN): path to the plain mips binary file containing
 * the UME+DAX core to be installed in the system.
 * @param iso (IN): path to the ISO9660 or DAX-ISO9660 file that will be 
 * used to emulate the UMD.
 * @param config (IN): pointer to the structure containing the 
 * configuration of the UME core.
 * @param daxconfig (IN): pointer to the structure containing the 
 * configuration of the DAX core.
 *
 * @return: -1 if the core cannot be installed (the file pointed
 * by "core" doesn't exist) or 0 if success.
 *
 * @Remarks:
 * If the file pointed by the parameter "core" is not a valid
 * UME+DAX core, the function won't return and the psp will crash
 * if an exception handler is not installed.
 * The same could happen if "core" is a valid UME+DAX core, but 
 * it's from an early version, since the compiler will probably
 * change the address of the daxCoreInit function.
 *
 * @Example of usage:
 * installUMEDAXCore("ms0:/UMDEMULATOR/FUSEDCORE.BIN", "ms0:/ISO/mygame.iso", 
   &myUMEcore_config, &myDAXcore_config);
 */
int installUMEDAXCore(char *core, char *iso, UMECoreConfig *config, DAXCoreConfig *daxconfig);

/*
 * It removes the UME core from the system.
*/
void uninstallUMECore();

/*
 * It removes the UMEDAX core from the system.
*/
void uninstallUMEDAXCore();



#endif /* _UMEDAX_H_ */
