/*
 * umedax.c (Dark_AleX)
 *
 * This file contains the implementation of functions to handle 
 * UME/DAX config files and to install/uninstall UME/UME+DAX cores.
*/

#include <pspsdk.h>
#include <pspkernel.h>
#include <pspiofilemgr.h>

#include <string.h>

#include "umedax.h"

u32 UME_DiscType2PSP_DiscType(u32 value)
{
	u32 array[3] =
	{
		PSP_DISCTYPE_GAME,
		PSP_DISCTYPE_VIDEO,
		PSP_DISCTYPE_AUDIO
	};

	return array[value];
}

u32 UME_Region2PSP_Region(u32 value)
{
	u32 array[4] =
	{
		PSP_REGION_JAPAN, 
		PSP_REGION_AMERICA,
		PSP_REGION_EUROPE,
		PSP_REGION_FREE
	};

	return array[value];
}

void defaultConfig(UMEDAXConfig *config)
{
	config->signature = UME_CONFIGFILE_SIGNATURE;
	config->disctypeISO = UME_DISCTYPE_GAME;
	config->disctypeUMD = UME_DISCTYPE_GAME;
	config->regiongame = UME_REGION_FREE;
	config->regionvideo = UME_REGION_FREE;
	config->regionvideo = UME_REGION_FREE;	

	config->coreconfig.speed = UME_SPEED_333;
	config->coreconfig.f0emu = 0;
	config->coreconfig.f1emu = 0;
	config->coreconfig.emulationmode = UME_EMUMODE_DIRECTLOAD;

	config->coreconfig.updateicon = UME_UPDATEICON_REMOVE;
	config->coreconfig.disctypeISO = PSP_DISCTYPE_GAME;
	config->coreconfig.disctypeUMD = PSP_DISCTYPE_GAME;
	config->coreconfig.usebootbin = 0;

	config->coreconfig.regcodeoverride = 0;
	config->coreconfig.regcodegame = PSP_REGION_FREE;
	config->coreconfig.regcodevideo = PSP_REGION_FREE;
	config->coreconfig.regcodeaudio = PSP_REGION_FREE;

	config->daxcoreconfig.emulationmode = DAX_EMUMODE_DIRECTLOAD;
	config->daxcoreconfig.allocatetables = 1;
}

void readConfig(char *file, UMEDAXConfig *config)
{
	SceUID fd;
	int size = 0;

	defaultConfig(config);

	fd = sceIoOpen(file, PSP_O_RDONLY, 0777);
	
	if (fd != -1)
	{
		size = sceIoRead(fd, config, sizeof(UMEDAXConfig));		
		sceIoClose(fd);
	}

	if (size < 4 || config->signature != UME_CONFIGFILE_SIGNATURE)
		defaultConfig(config);		

	// Check for invalid values to correct them
	if (config->coreconfig.speed > UME_SPEED_333)
		config->coreconfig.speed = UME_SPEED_333;

	if (config->coreconfig.f0emu > 1)
		config->coreconfig.f0emu = 1;

	if (config->coreconfig.f1emu > 1)
		config->coreconfig.f1emu = 1;

	if (config->coreconfig.emulationmode > UME_EMUMODE_COPYLOAD)
		config->coreconfig.emulationmode = UME_EMUMODE_DIRECTLOAD;
	
	if (config->coreconfig.updateicon > UME_UPDATEICON_REMOVE)
		config->coreconfig.updateicon = UME_UPDATEICON_REMOVE;

	if (config->disctypeISO > UME_DISCTYPE_AUDIO)
		config->disctypeISO = UME_DISCTYPE_GAME;

	if (config->disctypeUMD > UME_DISCTYPE_AUDIO)
		config->disctypeUMD = UME_DISCTYPE_GAME;	

	if (config->coreconfig.usebootbin > 1)
		config->coreconfig.usebootbin = 1;

	if (config->coreconfig.regcodeoverride > 1)
		config->coreconfig.regcodeoverride = 1;

	if (config->regiongame > UME_REGION_FREE)
		config->regiongame = UME_REGION_FREE;		

	if (config->regionvideo > UME_REGION_FREE)
		config->regionvideo = UME_REGION_FREE;		

	if (config->regionaudio > UME_REGION_FREE)
		config->regionaudio = UME_REGION_FREE;	

	if (config->daxcoreconfig.emulationmode > DAX_EMUMODE_SYSTEMMENURUNUMD)
		config->daxcoreconfig.emulationmode = config->coreconfig.emulationmode;
	
	if (config->daxcoreconfig.allocatetables > 1)
		config->daxcoreconfig.allocatetables = 1;

	config->coreconfig.disctypeISO = UME_DiscType2PSP_DiscType(config->disctypeISO);
	config->coreconfig.disctypeUMD = UME_DiscType2PSP_DiscType(config->disctypeUMD);
	config->coreconfig.regcodegame = UME_Region2PSP_Region(config->regiongame);
	config->coreconfig.regcodevideo = UME_Region2PSP_Region(config->regionvideo);
	config->coreconfig.regcodeaudio = UME_Region2PSP_Region(config->regionaudio);
}

void writeConfig(char *file, UMEDAXConfig *config)
{
	SceUID fd;

	fd = sceIoOpen(file, PSP_O_WRONLY | PSP_O_CREAT, 0777);

	if (fd >= 0)
	{
		sceIoWrite(fd, config, sizeof(UMEDAXConfig));
		sceIoClose(fd);
	}
}

int installUMECore(char *core, char *iso, UMECoreConfig *config)
{
	SceUID fd;
	UME_INIT_TYPE umeCoreInit;	

	fd = sceIoOpen(core, PSP_O_RDONLY, 0777);

	if (fd < 0)	
		return -1; 
	
	sceIoRead(fd, (void *)UME_CORE_ADDRESS, /* more than enough */0x0100000);
	sceIoClose(fd);

	strcpy((char *)UME_ISOSTRING_ADDRESS, iso);
	memcpy((void *)UME_CORECONFIG_ADDRESS, config, sizeof(UMECoreConfig));

	// Redirect system bootstrap
	// c0 88 01 3c lui $at, 0x88c0 -> 3e 88 01 3c lui $at, 0x883e
	// 09 f8 20 00 jalr $ra, $at
	_sh(0x883e, BOOTSTRAP_INST1);
	
	// The same here
	_sh(0x883e, BOOTSTRAP_INST2);

	// Call initialization code
	umeCoreInit = (void *)UME_INITCORE_ADDRESS;
	umeCoreInit();
	
	return 0;
}

int installUMEDAXCore(char *core, char *iso, UMECoreConfig *config, DAXCoreConfig *daxconfig)
{
	SceUID fd;
	UME_INIT_TYPE umeCoreInit;
	DAX_INIT_TYPE daxCoreInit;

	fd = sceIoOpen(core, PSP_O_RDONLY, 0777);

	if (fd < 0)	
		return -1; 
	
	sceIoRead(fd, (void *)UMEDAX_CORE_ADDRESS, /* more than enough */0x0100000);
	sceIoClose(fd);

	strcpy((char *)UME_ISOSTRING_ADDRESS, iso);
	memcpy((void *)UME_CORECONFIG_ADDRESS, config, sizeof(UMECoreConfig));

	// Redirect system bootstrap
	// c0 88 01 3c lui $at, 0x88c0 -> 3e 88 01 3c lui $at, 0x883e
	// 09 f8 20 00 jalr $ra, $at
	_sh(0x883e, BOOTSTRAP_INST1);

	// The same here
	_sh(0x883e, BOOTSTRAP_INST2);

	// Call initialization code
	daxCoreInit = (void *)DAX_INITCORE_ADDRESS;
	daxCoreInit(daxconfig);

	umeCoreInit = (void *)UME_INITCORE_ADDRESS;
	umeCoreInit();	
	
	return 0;
}

void uninstallUMECore()
{   
	// Restorer system bootstrap
   _sh(0x88c0, BOOTSTRAP_INST1);
   _sb(0xc0, 0x883e0430);
   _sh(0x88c0, BOOTSTRAP_INST2);
   _sb(0xc0, 0x883e0420);
}

void uninstallUMEDAXCore()
{
	// Nop repatch
	_sw(0, 0x883e0418);
	
	// Restorer system bootstrap
	_sh(0x88c0, BOOTSTRAP_INST1);
	_sh(0x88c0, BOOTSTRAP_INST2);
}
